/**************************************************************************
 **                                                                      **
 ** Copyright (C) 2011-2026 Lukas Spies                                  **
 ** Contact: https://photoqt.org                                         **
 **                                                                      **
 ** This file is part of PhotoQt.                                        **
 **                                                                      **
 ** PhotoQt is free software: you can redistribute it and/or modify      **
 ** it under the terms of the GNU General Public License as published by **
 ** the Free Software Foundation, either version 2 of the License, or    **
 ** (at your option) any later version.                                  **
 **                                                                      **
 ** PhotoQt is distributed in the hope that it will be useful,           **
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of       **
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        **
 ** GNU General Public License for more details.                         **
 **                                                                      **
 ** You should have received a copy of the GNU General Public License    **
 ** along with PhotoQt. If not, see <http://www.gnu.org/licenses/>.      **
 **                                                                      **
 **************************************************************************/

import QtQuick
import PhotoQt

Rectangle {

    id: minimap_top

    Item {
        id: containerItemForAnchors
        x: (parent.width-width)/2
        y: (parent.height-height)/2
        width: img.width+(PQCScriptsConfig.isQtAtLeast6_5() ? 0 : 6)
        height: img.height+(PQCScriptsConfig.isQtAtLeast6_5() ? 0 : 6)
    }

    x: PQCConstants.availableWidth-minimap_top.width-50
    y: PQCConstants.availableHeight-minimap_top.height-50
    // parent: parent
    width: img.width+(PQCScriptsConfig.isQtAtLeast6_5() ? 0 : 6)
    height: img.height+(PQCScriptsConfig.isQtAtLeast6_5() ? 0 : 6)
    opacity: minimap_top.minimapNeeded ? ((minimap_top.minimapActive||minimap_top.containsMouse) ? 1 : 0.2) : 0

    color: palette.base
    border.width: PQCScriptsConfig.isQtAtLeast6_5() ? 1 : 0
    border.color: PQCLook.baseBorder

    PQShadowEffect {
        masterItem: minimap_top
        z: PQCConstants.currentZValue
    }

    property bool minimapNeeded: (PQCConstants.currentImageScale > PQCConstants.currentImageDefaultScale*1.01 && (PQCConstants.currentVisibleContentSize.width > PQCConstants.availableWidth || PQCConstants.currentVisibleContentSize.height > PQCConstants.availableHeight)) && !PQCConstants.slideshowRunning && !PQCConstants.showingPhotoSphere
    property bool minimapActive: false
    property bool containsMouse: movemouse.containsMouse||minimapmouse.containsMouse||navmouse.containsMouse

    Connections {
        target: PQCConstants
        function onCurrentImageScaleChanged() {
            minimap_top.minimapActive = true
            minimapMakeDeactive.start()
        }
        function onCurrentVisibleAreaXChanged() {
            minimap_top.minimapActive = true
            minimapMakeDeactive.start()
        }
        function onCurrentVisibleAreaYChanged() {
            minimap_top.minimapActive = true
            minimapMakeDeactive.start()
        }
    }

    Timer {
        id: minimapMakeDeactive
        interval: 2000
        onTriggered: {
            minimap_top.minimapActive = false
        }
    }

    state: PQCSettings.interfaceMinimapPopout ? "popout" : "normal"

    z: PQCConstants.currentZValue


    Behavior on opacity { enabled: !PQCSettings.generalDisableAllAnimations; NumberAnimation { duration: 200 } }
    visible: opacity>0

    Text {
        color: palette.text
        anchors.centerIn: containerItemForAnchors
        visible: (img.source===""||img.status!=Image.Ready) && PQCSettings.interfaceMinimapPopout
        text: qsTranslate("image", "Minimap")
    }

    MouseArea {
        id: movemouse
        anchors.fill: containerItemForAnchors
        enabled: !PQCScriptsConfig.isQtAtLeast6_5()
        hoverEnabled: true
        cursorShape: PQCSettings.interfaceMinimapPopout ? Qt.ArrowCursor : Qt.SizeAllCursor
        drag.target: PQCSettings.interfaceMinimapPopout ? undefined : parent
    }

    MouseArea {
        id: minimapmouse
        anchors.fill: containerItemForAnchors
        anchors.margins: PQCScriptsConfig.isQtAtLeast6_5() ? 0 : 5
        hoverEnabled: true
        drag.target: PQCSettings.interfaceMinimapPopout ? undefined : parent
        acceptedButtons: Qt.AllButtons
        cursorShape: Qt.PointingHandCursor
        onClicked: (mouse) => {

            if(mouse.button === Qt.LeftButton) {

                var propX = mouse.x/img.width
                var propY = mouse.y/img.height

                PQCNotify.currentFlickableAnimateContentPosChange(propX,propY)

            } else if(mouse.button === Qt.RightButton) {
                rightclickmenu.popup()
            }
        }
        onWheel: (wheel) => {
            if(!PQCSettings.interfaceMinimapPopout) {
                wheel.accepted = false
                return
            }
            var pos = minimapmouse.mapToItem(toplevel, wheel.x, wheel.y)
            PQCNotify.mouseWheel(pos, wheel.angleDelta, wheel.modifiers)
        }


    }

    PQMenu {

        id: rightclickmenu

        PQMenuItem {
            text: qsTranslate("image", "Very small minimap")
            onTriggered:
            PQCSettings.imageviewMinimapSizeLevel = 0
        }

        PQMenuItem {
            text: qsTranslate("image", "Small minimap")
            onTriggered:
            PQCSettings.imageviewMinimapSizeLevel = 1
        }

        PQMenuItem {
            text: qsTranslate("image", "Normal minimap")
            onTriggered:
            PQCSettings.imageviewMinimapSizeLevel = 2
        }

        PQMenuItem {
            text: qsTranslate("image", "Large minimap")
            onTriggered:
            PQCSettings.imageviewMinimapSizeLevel = 3
        }

        PQMenuItem {
            text: qsTranslate("image", "Very large minimap")
            onTriggered:
            PQCSettings.imageviewMinimapSizeLevel = 4
        }

        PQMenuSeparator {}

        PQMenuItem {
            text: qsTranslate("image", "Hide minimap")
            onTriggered:
            PQCSettings.imageviewShowMinimap = false
        }

    }

    Image {
        id: img

        x: PQCScriptsConfig.isQtAtLeast6_5() ? 0 : 3
        y: PQCScriptsConfig.isQtAtLeast6_5() ? 0 : 3
        sourceSize: PQCSettings.imageviewMinimapSizeLevel === 4 ?
                        Qt.size(650, 650) :
                        (PQCSettings.imageviewMinimapSizeLevel === 3 ?
                            Qt.size(450,450) :
                            (PQCSettings.imageviewMinimapSizeLevel === 2 ?
                                 Qt.size(250, 250) :
                                 (PQCSettings.imageviewMinimapSizeLevel === 1 ?
                                      Qt.size(125,125) :
                                      Qt.size(75,75))))

        fillMode: Image.PreserveAspectFit
        source: ""
        cache: false
        asynchronous: true

        clip: true

        Rectangle {
            id: highlight_rect
            x: parent.width*PQCConstants.currentVisibleAreaX
            y: parent.height*PQCConstants.currentVisibleAreaY
            width: parent.width*PQCConstants.currentVisibleAreaWidthRatio
            height: parent.height*PQCConstants.currentVisibleAreaHeightRatio
            opacity: 0.7
            color: "#BB000000"
            border.width: 2
            border.color: "black"
            visible: minimap_top.minimapNeeded
            radius: 2
            MouseArea {
                id: navmouse
                anchors.fill: parent
                hoverEnabled: true
                drag.target: parent
                cursorShape: Qt.SizeAllCursor
                drag.onActiveChanged: {
                    if(!drag.active) {
                        highlight_rect.x = Qt.binding(function() { return highlight_rect.parent.width*PQCConstants.currentVisibleAreaX } )
                        highlight_rect.y = Qt.binding(function() { return highlight_rect.parent.height*PQCConstants.currentVisibleAreaY } )
                        PQCNotify.currentFlickableReturnToBounds()
                    } else {
                        highlight_rect.x = highlight_rect.x
                        highlight_rect.y = highlight_rect.y
                    }
                }
            }
            onXChanged: {
                if(navmouse.drag.active) {
                    PQCNotify.currentFlickableSetContentX(PQCConstants.currentVisibleContentSize.width*(x/img.width))
                }
            }
            onYChanged: {
                if(navmouse.drag.active) {
                    PQCNotify.currentFlickableSetContentY(PQCConstants.currentVisibleContentSize.height*(y/img.height))
                }
            }
        }

    }

    // This signal is emitted AFTER the image has been displayed. There might be some actions that happen in the meantime (e.g., very large archives
    // with very many files might sit for a while loading) and we don't want to bother with a mipmap until that is done
    Connections {
        target: PQCNotify
        function onNewImageHasBeenDisplayed() {
            img.source = encodeURI(PQCFileFolderModel.currentFile==="" ? "" : ("image://mipmap/" + PQCFileFolderModel.currentFile))
        }
    }

}
